﻿﻿/*
 * Copyright(C) 2015 SJNKS Corporation, All rights reserved.
 * $Id$
 * @fileoverview shomeiCommon.js 署名共通関数クラスのJavaScriptファイルです。
 * @author SJNKS 中川
 */

/**********************************
 * 定数
 * ********************************/
var defaultX = 600;					// 出力する画像サイズ（幅）
var defaultY = 200;					// 出力する画像サイズ（横）
var STU300 = {						// STU300
	WIDTH : 396,
	HEIGHT: 100,
	TYPE : 1
};
var STU430 = {						// STU430
	WIDTH : 320,
	HEIGHT: 200,
	TYPE : 2
};
//入力デバイス種別
var deviceType = {
	INK_TOOL        :"1",		// サインタブレット
	PEN_TABLET      :"2",		// ペンタブレット
	TOUCH_DEVICE    :"3",		// タブレット
	MOUSE           :"4"		// マウス
};
// ブラウザ種別
var browserType = {
	BROWSER_IOS     :"0",		// iOS
	BROWSER_ANDROID :"1",		// Android
	BROWSER_FIREFOX :"2",		// FireFox
	BROWSER_IE      :"3",		// IE
	BROWSER_CHROME  :"4",		// Chrome
	BROWSER_UNKNOWN :"99"		// 上記以外
};

/**********************************
 * 変数
 * ********************************/
var currentDevice = null;							// 現在使用中の入力デバイス
var currentBrowser = browserType.BROWSER_UNKNOWN;	// 現在使用中のブラウザ
var isTouchAvailable = false;						// タッチイベント使用可能フラグ
var isWebPluginAvailable = false;					// ウェブプラグイン使用可能フラグ
var isInkToolAvailable = false;					// InkTool使用可能フラグ
var isMouseAvailable = false;						// テスト用マウス使用可能フラグ
var isShomeiInput = false;							// 署名入力済みフラグ
var currentInkToolStatus = {
	dispMode : 1,		// 表示モード
	signTabletType : 0, // サインタブレット種別
	addOffset : 0,		// エリア限定時に使用するオフセット
	ratio : 1,			// 比率
	isInit : false		// 初期化フラグ
};

/**********************************
 * 署名共通関数
 * ********************************/
var shomeiCommon = {
	/**
	 * ユーザーエージェントを用いてブラウザの判定を行う。
	 */
	checkBrowser : function () {
		var lowerUA = navigator.userAgent.toLowerCase();
		if (0 < lowerUA.indexOf("ipad")) {
			// UAに「iPad」という文字が含まれていればiPad
			currentBrowser = browserType.BROWSER_IOS;
		} else if (0 < lowerUA.indexOf("android")) {
			// UAに「Android」という文字が含まれていればAndroid
			currentBrowser = browserType.BROWSER_ANDROID;
		} else if (0 < lowerUA.indexOf("firefox")) {
			// UAに「FireFox」という文字が含まれていればFireFox
			currentBrowser = browserType.BROWSER_FIREFOX;
		} else if (0 < lowerUA.indexOf("chrome")) {
			// UAに「chrome」という文字が含まれていればChrome
			currentBrowser = browserType.BROWSER_CHROME;
		} else if (0 < lowerUA.indexOf("msie") || 0 <lowerUA.indexOf("trident")) {
			// UAに「MSIE」（IE10以前）、または「Trident」（IE11）が含まれていればIE
			currentBrowser = browserType.BROWSER_IE;
		} else {
			// 対象外のブラウザ
			currentBrowser = browserType.BROWSER_UNKNOWN;
		}
	},

	/**
	 * 使用可能なデバイス数を取得する。
	 * @return {Number} 使用可能なデバイス数
	 */
	getDeviceCount : function () {
		var deviceCount = 0;

		// 各入力デバイスの使用前提条件を満たしているか判定する。
		if (shomeiCommon.checkCanvasAvailable()) {
			if (isTouchAvailable = shomeiCommon.checkTouchAvailable()) {
				deviceCount++;
			}
			if (isWebPluginAvailable = shomeiCommon.checkWebPluginAvailable()) {
				deviceCount++;
			}
		}
		if (isInkToolAvailable = shomeiCommon.checkInkToolAvailable()) {
			deviceCount++;
		}
		return deviceCount;
	},

	/**
	 * タッチイベントが使用可能かどうかを判定する。
	 * @return {boolean} true：使用可能、false：使用不可
	 */
	checkTouchAvailable : function () {
		var hasTapEvent = (function() {
			var div = document.createElement('div');
			div.setAttribute('ontouchstart', 'return');
			return (typeof div.ontouchstart === 'function');
		})();
		if (hasTapEvent) {
			// iOS、Androidの場合
			return true;
		} else if (navigator.maxTouchPoints && navigator.maxTouchPoints > 0) {
			// Windowsタブレットの場合
			return true;
		} else {
			return false;
		}
	},

	/**
	 * WebPluginが使用可能かどうかを判定する。
	 * @return {boolean} true：使用可能、false：使用不可
	 */
	checkWebPluginAvailable : function () {
		try {
			if ($("#wtPlugin")!=null && $("#wtPlugin")[0].penAPI) {
				return true;
			} else {
				return false;
			}
		} catch (ex) {
			// WebPluginオブジェクトのプロパティ取得できないときは例外が発生する。
			return false;
		}
	},

	/**
	 * Canvasが使用可能かどうかを判定する。
	 * @return {boolean} true：使用可能、false：使用不可
	 */
	checkCanvasAvailable : function () {
		try {
			document.createElement("canvas").getContext("2d");
			return true;
		} catch (ex) {
			// canvasが使用できない場合は例外が発生する。
			return false;
		}
	},

	/**
	 * InkToolが利用可能かどうかを判定する。
	 * @return {boolean} true：使用可能、false：使用不可
	 */
	checkInkToolAvailable : function () {
		try {
//			if ($("#InkTool")[0].PenWidth && shomeiInkTool.isSignPadConnect($("#InkTool")[0]) > -1) {
			if ($("#InkTool")[0].PenWidth ) {
				return true;
			} else {
				return false;
			}
		} catch (ex) {
			// 作成できない場合は例外が発生する。
			return false;
		}
	},

	/**
	 * Imageデータから署名データを復元する
	 * @param {String} shomeiData 復元する署名データ（base64文字列）
	 */
	restore : function (shomeiData) {
		var mainCanvas = shomeiUtil.mainCanvasObj[0];
		// Imageオブジェクトを作成
		imageData = new Image();
		imageData.src = "data:image/png;base64," + shomeiData;

		// 画像のロードが終わるまで待つ
		imageData.onload = function () {
			// 描画の前にcanvasをクリア
			shomeiCommon.clear();
			
			// メイン画面のcanvasに復元
			mainCanvas.getContext("2d").drawImage(imageData, 0, 0, defaultX, defaultY);
			// 署名入力済みフラグを設定
			isShomeiInput = true;
		};
	},

	/**
	 * 署名データを文字列に変換（Base64エンコード）して出力する。
	 * @return {String} 画像データをbase64エンコードした文字列
	 */
	output : function () {
		var mainCanvas = shomeiUtil.mainCanvasObj[0];
		return mainCanvas.toDataURL().replace("data:image/png;base64,", "");
	},

	/**
	 * 署名画面のcanvasをクリアする。
	 */
	clear : function () {
		// canvasの入力をクリア
		var ctx = shomeiUtil.mainCanvasObj[0].getContext("2d");
		ctx.clearRect(0, 0, shomeiUtil.mainCanvasObj.width, shomeiUtil.mainCanvasObj.height);

		// 背景を白で塗りつぶす
		ctx.fillStyle = "White";
		ctx.fillRect(0, 0, defaultX, defaultY);
	},

	/** 入力デバイス種別を保持するクッキーのキー値 */
	COOKIE_DEVICE_KEY : "deviceType",

	/**
	 * ユーザーが選択した入力デバイス種別を取得する。
	 * @return {deviceType} ユーザーが選択した入力デバイス種別
	 */
	getDeviceType : function () {
		var value = cookie_read(shomeiCommon.COOKIE_DEVICE_KEY);
		if (value == null || value == "undefined") {
			return "";
		}
		return value;
	},

	/**
	 * ユーザーが選択した入力デバイス種別を設定する。
	 * 設定先は cookie_device_keyをキー値とするクッキー。有効期限は 365日 (31536000秒)
	 * @param {deviceType} selectedDevice ユーザーが選択した入力デバイス種別
	 */
	setDeviceType : function (selectedDevice) {
		var cookiePath = make_PATH();
		var expire = new Date();
		expire.setTime( expire.getTime() + 1000 * 3600 * 24 * 365);
		var expires = expire.toUTCString(); 
		document.cookie = shomeiCommon.COOKIE_DEVICE_KEY + "=" + selectedDevice + ";Path=" + cookiePath + ";expires=" + expires;
	},

	/**
	 * ユーザーが選択した入力デバイス種別を削除する。
	 */
	removeDeviceType : function () {
		shomeiCommon.setDeviceType("");
	}
};