/*
 * Copyright(C) 2015 SCSK Corporation, All rights reserved.
 * $Id$
 * @fileoverview screenCommon 画面共通処理
 * @author SCSK
*/

/*
 * 概要：ページトップへ戻る
 */
paperless.event.click(null, ".page-top", function (ev) {
	$("body,html").animate({ scrollTop: 0 }, 200);
	return false;
});

/*
 * 概要：href="#"でのページトップへの移動を無効にする
 */
paperless.event.click(null, "a[href=#]", function (ev) {
	var $this = $(this);
	if ($this.hasClass("page-top") || $this.attr("data-rel")) return;
	ev.preventDefault();
});

/*
 * 正規表現にマッチする文字を指定コード分移動した文字に変換します。
 *
 * @param  {string} expr  対象文字列
 * @param  {string} regex 変換対象の文字を判別する正規表現
 * @param  {string} move  文字コード移動量
 * @return {string} 変換後の文字列
 */
function toReverseCode( expr, regex, move ) {
	return expr.replace( regex , function( str ) {
		return String.fromCharCode( str.charCodeAt( 0 ) - move );
	});
};

/*
 * 半角カタカナを全角カタカナに変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function hankakuToZenkakuKana ( expr ) {
	var before = [ "ｶﾞ","ｷﾞ","ｸﾞ","ｹﾞ","ｺﾞ","ｻﾞ","ｼﾞ","ｽﾞ","ｾﾞ","ｿﾞ","ﾀﾞ","ﾁﾞ","ﾂﾞ","ﾃﾞ","ﾄﾞ","ﾊﾞ","ﾋﾞ","ﾌﾞ","ﾍﾞ","ﾎﾞ","ﾊﾟ","ﾋﾟ","ﾌﾟ","ﾍﾟ","ﾎﾟ","ｦ","ｧ","ｨ","ｩ","ｪ","ｫ","ｬ","ｭ","ｮ","ｯ","ｰ","ｱ","ｲ","ｳ","ｴ","ｵ","ｶ","ｷ","ｸ","ｹ","ｺ","ｻ","ｼ","ｽ","ｾ","ｿ","ﾀ","ﾁ","ﾂ","ﾃ","ﾄ","ﾅ","ﾆ","ﾇ","ﾈ","ﾉ","ﾊ","ﾋ","ﾌ","ﾍ","ﾎ","ﾏ","ﾐ","ﾑ","ﾒ","ﾓ","ﾔ","ﾕ","ﾖ","ﾗ","ﾘ","ﾙ","ﾚ","ﾛ","ﾜ","ﾝ" ];
	var after  = [ "ガ","ギ","グ","ゲ","ゴ","ザ","ジ","ズ","ゼ","ゾ","ダ","ヂ","ヅ","デ","ド","バ","ビ","ブ","ベ","ボ","パ","ピ","プ","ペ","ポ","ヲ","ァ","ィ","ゥ","ェ","ォ","ャ","ュ","ョ","ッ","ー","ア","イ","ウ","エ","オ","カ","キ","ク","ケ","コ","サ","シ","ス","セ","ソ","タ","チ","ツ","テ","ト","ナ","ニ","ヌ","ネ","ノ","ハ","ヒ","フ","ヘ","ホ","マ","ミ","ム","メ","モ","ヤ","ユ","ヨ","ラ","リ","ル","レ","ロ","ワ","ン" ];
	var regex  = /[ｦ-ﾝ]/;
	return toAfterArray( expr, before, after, regex );
};

/*
 * NKL証券番号の全角カタカナを半角カタカナに変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function zenkakuToHankakuKanaShokenNKL ( expr ) {
	var before = [ "ヲ","ァ","ィ","ゥ","ェ","ォ","ャ","ュ","ョ","ッ","ー","ア","イ","ウ","エ","オ","カ","キ","ク","ケ","コ","サ","シ","ス","セ","ソ","タ","チ","ツ","テ","ト","ナ","ニ","ヌ","ネ","ノ","ハ","ヒ","フ","ヘ","ホ","マ","ミ","ム","メ","モ","ヤ","ユ","ヨ","ラ","リ","ル","レ","ロ","ワ","ン" ];
	var after  = [ "ｦ","ｧ","ｨ","ｩ","ｪ","ｫ","ｬ","ｭ","ｮ","ｯ","ｰ","ｱ","ｲ","ｳ","ｴ","ｵ","ｶ","ｷ","ｸ","ｹ","ｺ","ｻ","ｼ","ｽ","ｾ","ｿ","ﾀ","ﾁ","ﾂ","ﾃ","ﾄ","ﾅ","ﾆ","ﾇ","ﾈ","ﾉ","ﾊ","ﾋ","ﾌ","ﾍ","ﾎ","ﾏ","ﾐ","ﾑ","ﾒ","ﾓ","ﾔ","ﾕ","ﾖ","ﾗ","ﾘ","ﾙ","ﾚ","ﾛ","ﾜ","ﾝ" ];
	var regex  = /[ァ-ヶ]/;
	return toAfterArray( expr, before, after, regex );
};

/*
 * 半角数字を全角数字に変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function hankakuToZenkakuSuji ( expr ) {
	var before = [ "0","1","2","3","4","5","6","7","8","9" ];
	var after  = [ "０","１","２","３","４","５","６","７","８","９" ];
	var regex  = /[0-9]/;
	return toAfterArray( expr, before, after, regex );
};

/*
 * 全角数字を半角数字に変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function zenkakuSujiToHankaku ( expr ) {
	var before  = [ "０","１","２","３","４","５","６","７","８","９" ];
	var after = [ "0","1","2","3","4","5","6","7","8","9" ];
	var regex  = /[０-９]/;
	return toAfterArray( expr, before, after, regex );
};

/*
 * 全角英数字を半角英数字に変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function toHankakuAlphabet ( expr ) {
	return toReverseCode( expr, /[０-９ａ-ｚＡ-Ｚ]/g, 0xFEE0 );
};

/*
 * 半角アルファベットを全角アルファベットに変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function toZenkakuAlphabet ( expr ) {
	return toMoveCode( expr, /[a-zA-Z]/g, 0xFEE0 );
};

/*
 * 半角スペースを全角スペースに変換します。
 *
 * @param  {string} expr 対象文字列
 * @return {string} 変換後の文字列
 */
function hankakuToZenkakuSpace ( expr ) {
	return expr.replace(/ /g,"　");
};

/*
 * 正規表現にマッチする文字を指定コード分移動した文字に変換します。
 *
 * @param  {string} expr  対象文字列
 * @param  {string} regex 変換対象の文字を判別する正規表現
 * @param  {string} move  文字コード移動量
 * @return {string} 変換後の文字列
 */
function toMoveCode ( expr, regex, move ) {
	return expr.replace( regex , function( str ) {
		return String.fromCharCode( str.charCodeAt( 0 ) + move );
	});
};

/*
 * 文字列の全置換を行います。
 *
 * @param  {string} expr 対象文字列
 * @param  {string} find 変換する文字列
 * @param  {string} dest 変換後の文字列
 * @return {string} 変換後の文字列
 */
function replaceAll( expr, find, dest ) {
	return expr.split( find ). join( dest );
};

/*
 * 配列内の文字列を変換対象配列内の文字列に変換します。
 *
 * @param  {string} expr   対象文字列
 * @param  {string} before 変換対象の文字列を格納した配列
 * @param  {string} after  変換後の文字列を格納した配列
 * @param  {string} regex  変換対象の文字を判別する正規表現
 * @return {string} 変換後の文字列
 */
function toAfterArray ( expr, before, after, regex ) {
	if ( expr.match( regex ) ) {
		for ( var i = 0; i < before.length; i++ ) {
			expr = replaceAll( expr, before[ i ], after[ i ] );
		}
	}
	return expr;
};

/*
 * idvDataから検索文字の値を取得します。
 *
 * @param {string} idvData idvData
 * @param {string} search 検索文字列
 */
function searchIdvDataValue(idvData, search) {
	for (var i = 0; i < idvData.length; i++) {
		var data = idvData[i];
		for (var j = 0; j < data.length; j++) {
			if (data[j].key == search) {
				return data[j].value;
			}
		}
	}
	return "";
}

/*
 * 概要：画面遷移（別HTML）[前画面に戻る]ボタン押下処理
 * @param {string}   null(ボタンID)
 * @param {string}   クラス名
 * @param {function} ボタン押下処理
 */
paperless.event.click(null, ".page-back-pl", function (ev) {
	history.back();
//	paperless.page.change("back", variable.bean, variable.id);
});

/*
 * 概要：画面遷移（別HTML）[メニュー・ヘルプ]ボタン押下処理
 * @param {string}   ボタンID
 * @param {function} ボタン押下処理
 */
paperless.event.click("b_menuHelp", function (ev) {
	clickMenuHelp();
});

/*
 * 概要：画面遷移（別HTML） [確認]リンク押下処理
 * @param {string} 証券番号
 */
function clickKakunin(shokenNo) {
	alert('一般照会画面に遷移します。');
	return;

	// 契約照会画面のウィンドウが裏側に回ることを避けるため、無理やりフォーカスしている。（代理店システムの動作仕様を引き継ぐ）
	var subWinObj = window.opener.open("","HOIPPAN","top=0,left=0,width=1014px,height=690px,scrollbars=1,status=1,resizable=1");
	subWinObj.blur();
	window.focus();
	window.blur();
	subWinObj.focus();
	// クライアント基盤・共通項目を取得
	var commonData = getCommonData();
	// 契約照会画面が半角カナを受け取るとエラーとなるため、半角カナ→全角カナ変換を行う（NKL証券番号対応）
	// 契約照会画面がNKL証券番号の枝番のハイフン（例：12345678901-1）を対応できないため、ハイフンを削除（HL契約のハイフン（例： 12345-678）は対応しているとのこと）
	var srcShoban = hankakuToZenkakuKana(shokenNo.replace(/-/g, ""));
	// 契約照会画面はEUCで動作するため、charsetを変更
	document.charset='EUC-JP';
	// ポスト先URLはPL解約システムのウィンドウの呼び出し元であるトップメニューから取得（PL解約システムでは契約照会画面のURLを持っていない）
	var HOIPPANForm = $("<form/>",{ target : "HOIPPAN", method : "post" , action : window.opener.$("#logoffFrm").attr("action") });
	HOIPPANForm
	.append(hidden("__sid", commonData.sessionId))
	.append(hidden("__uid", commonData.userId))
	.append(hidden("__func", "一般照会"))
	.append(hidden("__gid", variable.id))
	.append(hidden("DBFlg", "0"))
	.append(hidden("__fid", "APL-101"))
	.append(hidden("__type", "0003"))
	.append(hidden("SrcShoban", srcShoban))
	.appendTo(document.body)
	.submit()
	.remove()
	;
	// PL解約システムの画面のcharsetへ戻す
	document.charset='UTF-8';
};

/* hiddenを生成して返す */
function hidden( name, value ) {
	return $( "<input/>" ).attr({
		type: "hidden", lang: "ja", name: name, value: value
	});
};

/*
 * 対象の文字列に0を先頭に追加（padding）します。
 *
 * @param  {string} expr 対象文字列
 * @param  {int} size 文字列長
 * @return {string} 変換後の文字列
 */
function paddingZero( expr, size ) {
	return paddingLeft(expr, "0", size);
};

/*
 * 対象の文字列に指定した追加文字列を先頭に追加（padding）します。
 *
 * @param  {string} expr 対象文字列
 * @param  {string} pad 追加文字列
 * @param  {int} size 文字列長
 * @return {string} 変換後の文字列
 */
function paddingLeft( expr, pad, size ) {
	var count = 0;
	var prefix = "";
	while (count < parseInt(size)) {
		prefix += pad;
		count++;
	}
	return (prefix + expr).slice(-size);
};

/*
 * メニューから表示したダイアログが閉じるとき、メニューも一緒にCloseする。
 *
 */
function closeOverlayWithMenu() {
	paperless.page.closeOverlay();
	setTimeout(function () {
		if ($("#menu").is(":visible")) {
			menuHelp.menuHelpObj.popup("close");
		}
	}, 100);
};