/**
 * 共通ユーティリティ
 *
 * @version 1.0
 * @author  scsk
 */
;$( function () {

	/*
	 * 文字列ユーティリティ
	 */
	StringUtil = {

		/*
		 * 文字列の全置換を行います。
		 *
		 * @param  {string} expr 対象文字列
		 * @param  {string} find 変換する文字列
		 * @param  {string} dest 変換後の文字列
		 * @return {string} 変換後の文字列
		 */
		replaceAll: function ( expr, find, dest ) {
			return expr.split( find ). join( dest );
		},

		/*
		 * 配列内の文字列を変換対象配列内の文字列に変換します。
		 *
		 * @param  {string} expr   対象文字列
		 * @param  {string} before 変換対象の文字列を格納した配列
		 * @param  {string} after  変換後の文字列を格納した配列
		 * @param  {string} regex  変換対象の文字を判別する正規表現
		 * @return {string} 変換後の文字列
		 */
		toAfterArray: function ( expr, before, after, regex ) {
			if ( expr.match( regex ) ) {
				for ( var i = 0; i < before.length; i++ ) {
					expr = StringUtil.replaceAll( expr, before[ i ], after[ i ] );
				}
			}
			return expr;
		},

		/*
		 * 正規表現にマッチする文字を指定コード分移動した文字に変換します。
		 *
		 * @param  {string} expr  対象文字列
		 * @param  {string} regex 変換対象の文字を判別する正規表現
		 * @param  {string} move  文字コード移動量
		 * @return {string} 変換後の文字列
		 */
		toMoveCode: function ( expr, regex, move ) {
			return expr.replace( regex , function( str ) {
				return String.fromCharCode( str.charCodeAt( 0 ) + move );
			});
		},

		/*
		 * 半角カタカナを全角カタカナに変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		hankakuToZenkakuKana: function ( expr ) {
			var before = [ "ｶﾞ","ｷﾞ","ｸﾞ","ｹﾞ","ｺﾞ","ｻﾞ","ｼﾞ","ｽﾞ","ｾﾞ","ｿﾞ","ﾀﾞ","ﾁﾞ","ﾂﾞ","ﾃﾞ","ﾄﾞ","ﾊﾞ","ﾋﾞ","ﾌﾞ","ﾍﾞ","ﾎﾞ","ﾊﾟ","ﾋﾟ","ﾌﾟ","ﾍﾟ","ﾎﾟ","ｦ","ｧ","ｨ","ｩ","ｪ","ｫ","ｬ","ｭ","ｮ","ｯ","ｰ","ｱ","ｲ","ｳ","ｴ","ｵ","ｶ","ｷ","ｸ","ｹ","ｺ","ｻ","ｼ","ｽ","ｾ","ｿ","ﾀ","ﾁ","ﾂ","ﾃ","ﾄ","ﾅ","ﾆ","ﾇ","ﾈ","ﾉ","ﾊ","ﾋ","ﾌ","ﾍ","ﾎ","ﾏ","ﾐ","ﾑ","ﾒ","ﾓ","ﾔ","ﾕ","ﾖ","ﾗ","ﾘ","ﾙ","ﾚ","ﾛ","ﾜ","ﾝ" ];
			var after  = [ "ガ","ギ","グ","ゲ","ゴ","ザ","ジ","ズ","ゼ","ゾ","ダ","ヂ","ヅ","デ","ド","バ","ビ","ブ","ベ","ボ","パ","ピ","プ","ペ","ポ","ヲ","ァ","ィ","ゥ","ェ","ォ","ャ","ュ","ョ","ッ","ー","ア","イ","ウ","エ","オ","カ","キ","ク","ケ","コ","サ","シ","ス","セ","ソ","タ","チ","ツ","テ","ト","ナ","ニ","ヌ","ネ","ノ","ハ","ヒ","フ","ヘ","ホ","マ","ミ","ム","メ","モ","ヤ","ユ","ヨ","ラ","リ","ル","レ","ロ","ワ","ン" ];
			var regex  = /[ｦ-ﾝ]/;
			return StringUtil.toAfterArray( expr, before, after, regex );
		},

		/*
		 * 半角数字を全角数字に変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		hankakuToZenkakuSuji: function ( expr ) {
			var before = [ "0","1","2","3","4","5","6","7","8","9" ];
			var after  = [ "０","１","２","３","４","５","６","７","８","９" ];
			var regex  = /[0-9]/;
			return StringUtil.toAfterArray( expr, before, after, regex );
		},

		/*
		 * 全角数字を半角数字に変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		zenkakuSujiToHankaku: function ( expr ) {
			var before  = [ "０","１","２","３","４","５","６","７","８","９" ];
			var after = [ "0","1","2","3","4","5","6","7","8","9" ];
			var regex  = /[０-９]/;
			return toAfterArray( expr, before, after, regex );
		},

		/*
		 * 半角記号を全角記号に変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		hankakuToZenkakuKigo: function ( expr ) {
			var before = [ "\\","&","%","?","!","*",".","<",">","-",'"',"#","$","'","=","|","[","]",":",";","^","/","､","｡","･","+","｣","｢","{","}","_",","," " ];
			var after  = [ "￥","＆","％","？","！","＊","．","＜","＞","－","”","＃","＄","’","＝","｜","［","］","：","；","＾","／","、","。","・","＋","」","「","｛","｝","＿","，","　" ];
			var regex  = /[\\\\&%\?!\*\.<>\-"#$'=\|\[\]:;\^/､｡･\+｣｢\{\}_, ]/;
			return StringUtil.toAfterArray( expr, before, after, regex );
		},

		/*
		 * ひらがなを全角カタカナに変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		hiraganaToZenkakuKana: function ( expr ) {
			return StringUtil.toMoveCode( expr, /[ぁ-ゔ]/g, 0x60 );
		},

		/*
		 * 半角アルファベットを全角アルファベットに変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		toZenkakuAlphabet: function ( expr ) {
			return StringUtil.toMoveCode( expr, /[a-zA-Z]/g, 0xFEE0 );
		},
		

		/*
		 * 全角小文字アルファベットを全角大文字アルファベットに変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		toZenkakuUpperAlphabet: function ( expr ) {
			return StringUtil.toMoveCode( expr, /[ａ-ｚ]/g, -0x20 );
		},

		/*
		 * カタカナ小文字をカタカナ大文字に変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		toUpperZenkakuKana: function ( expr ) {
			var before = [ "ァ","ィ","ゥ","ェ","ォ","ッ","ャ","ュ","ョ","ヮ","ヵ","ヶ" ];
			var after  = [ "ア","イ","ウ","エ","オ","ツ","ヤ","ユ","ヨ","ワ","カ","ケ" ];
			var regex  = /[ァィゥェォッャュョヮヵヶ]/;
			return StringUtil.toAfterArray( expr, before, after, regex );
		},
		
		

		/*
		 * その他の全角カタカナ「ヲー」を「オ－」に変換します。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		toOtherZenkakuKana : function ( expr ) {
			var before = [ "ヲ","ー" ];
			var after  = [ "オ","－" ];
			var regex  = /[ヲー]/;
			return StringUtil.toAfterArray( expr, before, after, regex );
		},

		/*
		 * 数値の場合前ゼロを取ります。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		suppressZero : function ( expr ) {
			var obj = Number( expr );
			if (isNaN( obj )) {
				return expr;
			}
			return String(obj);
		},

		/*
		 * 改行を取り除きます。
		 *
		 * @param  {string} expr 対象文字列
		 * @return {string} 変換後の文字列
		 */
		suppressReturn : function ( expr ) {
			return expr.replace(/[\t\r\n\f]/g,"");
		}
	};

	/* PageUtil */
	PageUtil = {

		/* 強制ダウンロード対象のブラウザ判定処理 */
		isPdfDonwloadBrowser: function() {
			// N-GW経由、F-GW経由のChromiumベースブラウザが対象
			if (location.href.match(/\/smtsso\/fw\//) || location.href.match(/\/sso\/fw\//)) {
				// PC以外で"chrome"が含まれるかどうか
				return paperless.browser.android.chrome;
			}
			return false;
		},

		/* PDFダウンロード（アンカー方式）*/
		pdfDownloader: function ( url ) {
			var html = '<!DOCTYPE html><html lang="ja"><head><meta charset="utf-8" /><meta name="viewport" content="width=980" /><title>ひまわりオンライン - PDFダウンロード</title></head><body><p>ひまわりオンラインでPDFファイルをダウンロードしています。</p><p>PDFファイルのダウンロードが終了しましたら、このタブを閉じてください。</p>※ダウンロードが開始されない場合は<a id="ancid" href="' + url + '" download>こちら</a>をタップしてください。</body></html>';
			var script = document.createElement("script");
			script.innerHTML = 'setTimeout(function(){document.getElementById("ancid").click();},500);';
			var pdfwin = window.open();
			pdfwin.document.open();
			pdfwin.document.write(html);
			pdfwin.document.body.appendChild(script);
			pdfwin.document.close();
		}
	};
});
